/*
 * Copyright (c) 2019 Yubico AB. All rights reserved.
 * Use of this source code is governed by a BSD-style
 * license that can be found in the LICENSE file.
 */

#include <assert.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "mutator_aux.h"
#include "fido.h"

#include "../openbsd-compat/openbsd-compat.h"

#define TAG_U2F		0x01
#define TAG_TYPE	0x02
#define TAG_CDH		0x03
#define TAG_RP_ID	0x04
#define TAG_RP_NAME	0x05
#define TAG_USER_ID	0x06
#define TAG_USER_NAME	0x07
#define TAG_USER_NICK	0x08
#define TAG_USER_ICON	0x09
#define TAG_EXT		0x0a
#define TAG_SEED	0x0b
#define TAG_RK		0x0c
#define TAG_UV		0x0d
#define TAG_PIN		0x0e
#define TAG_WIRE_DATA	0x0f
#define TAG_EXCL_COUNT	0x10
#define TAG_EXCL_CRED	0x11

/* Parameter set defining a FIDO2 make credential operation. */
struct param {
	char		pin[MAXSTR];
	char		rp_id[MAXSTR];
	char		rp_name[MAXSTR];
	char		user_icon[MAXSTR];
	char		user_name[MAXSTR];
	char		user_nick[MAXSTR];
	int		ext;
	int		seed;
	struct blob	cdh;
	struct blob	excl_cred;
	struct blob	user_id;
	struct blob	wire_data;
	uint8_t		excl_count;
	uint8_t		rk;
	uint8_t		type;
	uint8_t		u2f;
	uint8_t		uv;
};

/* Example parameters. */
static const char dummy_rp_id[] = "localhost";
static const char dummy_rp_name[] = "sweet home localhost";
static const char dummy_pin[] = "9}4gT:8d=A37Dh}U";
static const char dummy_user_icon[] = "an icon";
static const char dummy_user_name[] = "john smith";
static const char dummy_user_nick[] = "jsmith";

static const uint8_t dummy_cdh[] = {
	0xf9, 0x64, 0x57, 0xe7, 0x2d, 0x97, 0xf6, 0xbb,
	0xdd, 0xd7, 0xfb, 0x06, 0x37, 0x62, 0xea, 0x26,
	0x20, 0x44, 0x8e, 0x69, 0x7c, 0x03, 0xf2, 0x31,
	0x2f, 0x99, 0xdc, 0xaf, 0x3e, 0x8a, 0x91, 0x6b,
};

static const uint8_t dummy_user_id[] = {
	0x78, 0x1c, 0x78, 0x60, 0xad, 0x88, 0xd2, 0x63,
	0x32, 0x62, 0x2a, 0xf1, 0x74, 0x5d, 0xed, 0xb2,
	0xe7, 0xa4, 0x2b, 0x44, 0x89, 0x29, 0x39, 0xc5,
	0x56, 0x64, 0x01, 0x27, 0x0d, 0xbb, 0xc4, 0x49,
};

/*
 * Collection of HID reports from an authenticator issued with a FIDO2
 * make credential using the example parameters above.
 */
static const uint8_t dummy_wire_data_fido[] = {
	0xff, 0xff, 0xff, 0xff, 0x86, 0x00, 0x11, 0xb0,
	0x84, 0xeb, 0xec, 0x4d, 0x97, 0x72, 0x09, 0x00,
	0x91, 0x00, 0x03, 0x02, 0x05, 0x00, 0x02, 0x05,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0x90, 0x00, 0x51, 0x00,
	0xa1, 0x01, 0xa5, 0x01, 0x02, 0x03, 0x38, 0x18,
	0x20, 0x01, 0x21, 0x58, 0x20, 0x69, 0xf2, 0x7d,
	0x37, 0x57, 0xda, 0x11, 0xba, 0x42, 0xde, 0x79,
	0xe4, 0xab, 0x8d, 0x73, 0x63, 0xee, 0x66, 0x9e,
	0x8a, 0x70, 0xa9, 0xb5, 0xf6, 0x38, 0x4f, 0x5b,
	0xdf, 0xe1, 0xa0, 0xa4, 0xff, 0x22, 0x58, 0x20,
	0x8a, 0xcb, 0x23, 0x2e, 0x93, 0xdb, 0xe0, 0xa4,
	0x00, 0x91, 0x00, 0x03, 0x00, 0xbb, 0xb5, 0x60,
	0x19, 0x18, 0x8b, 0x4d, 0xb8, 0x88, 0x6e, 0x13,
	0x75, 0xac, 0x00, 0x19, 0x27, 0x80, 0xcc, 0x63,
	0xc4, 0xbf, 0xfe, 0x4b, 0x4a, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0x90, 0x00, 0x14, 0x00,
	0xa1, 0x02, 0x50, 0x10, 0x89, 0x77, 0x43, 0x3a,
	0x58, 0xa2, 0xc9, 0x98, 0x18, 0x1a, 0xb1, 0xcc,
	0x09, 0x6b, 0xd0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0xbb, 0x00, 0x01, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x91, 0x00, 0x03, 0x90, 0x03, 0xe1, 0x00,
	0xa3, 0x01, 0x66, 0x70, 0x61, 0x63, 0x6b, 0x65,
	0x64, 0x02, 0x58, 0xc4, 0x49, 0x96, 0x0d, 0xe5,
	0x88, 0x0e, 0x8c, 0x68, 0x74, 0x34, 0x17, 0x0f,
	0x64, 0x76, 0x60, 0x5b, 0x8f, 0xe4, 0xae, 0xb9,
	0xa2, 0x86, 0x32, 0xc7, 0x99, 0x5c, 0xf3, 0xba,
	0x83, 0x1d, 0x97, 0x63, 0x45, 0x00, 0x00, 0x00,
	0x00, 0xf8, 0xa0, 0x11, 0xf3, 0x8c, 0x0a, 0x4d,
	0x00, 0x91, 0x00, 0x03, 0x00, 0x15, 0x80, 0x06,
	0x17, 0x11, 0x1f, 0x9e, 0xdc, 0x7d, 0x00, 0x40,
	0xed, 0x88, 0x48, 0xa1, 0xdb, 0x56, 0x4d, 0x0f,
	0x0d, 0xc8, 0x8f, 0x0f, 0xe9, 0x16, 0xb1, 0x78,
	0xa9, 0x40, 0x98, 0x71, 0xa0, 0xb3, 0xf2, 0xcf,
	0x05, 0x73, 0x6c, 0x12, 0xbf, 0x00, 0x96, 0xf3,
	0x7b, 0x93, 0xba, 0x49, 0xee, 0x23, 0xb4, 0x78,
	0x2e, 0xfb, 0xce, 0x27, 0xa8, 0xc2, 0x26, 0x78,
	0x00, 0x91, 0x00, 0x03, 0x01, 0xcc, 0x95, 0x2d,
	0x40, 0xdb, 0xd1, 0x40, 0x3d, 0x2b, 0xa3, 0x31,
	0xa0, 0x75, 0x82, 0x63, 0xf0, 0xa5, 0x01, 0x02,
	0x03, 0x26, 0x20, 0x01, 0x21, 0x58, 0x20, 0x9d,
	0x95, 0xa1, 0xb5, 0xd6, 0x11, 0xbf, 0xe2, 0x28,
	0xa0, 0x7f, 0xca, 0x1e, 0xd9, 0x09, 0x0f, 0x0d,
	0xe7, 0x8e, 0x29, 0xe8, 0x2e, 0x11, 0xdb, 0x55,
	0x62, 0x13, 0xd7, 0x26, 0xc2, 0x7e, 0x2b, 0x22,
	0x00, 0x91, 0x00, 0x03, 0x02, 0x58, 0x20, 0xbe,
	0x74, 0x2a, 0xac, 0xde, 0x11, 0x40, 0x76, 0x31,
	0x0b, 0xed, 0x55, 0xde, 0xf3, 0x03, 0xe4, 0x1c,
	0xac, 0x42, 0x63, 0x8f, 0xe8, 0x30, 0x63, 0xb7,
	0x07, 0x4e, 0x5d, 0xfb, 0x17, 0x5e, 0x9b, 0x03,
	0xa3, 0x63, 0x61, 0x6c, 0x67, 0x26, 0x63, 0x73,
	0x69, 0x67, 0x58, 0x48, 0x30, 0x46, 0x02, 0x21,
	0x00, 0xfb, 0xd1, 0x26, 0x76, 0x34, 0x74, 0xac,
	0x00, 0x91, 0x00, 0x03, 0x03, 0xf6, 0xd8, 0x5c,
	0x5d, 0xbc, 0xda, 0xe0, 0x43, 0xe0, 0xa5, 0x42,
	0x9f, 0xc7, 0xe2, 0x18, 0x3e, 0xe2, 0x2c, 0x94,
	0x78, 0xbf, 0x9c, 0xeb, 0x3e, 0x9d, 0x02, 0x21,
	0x00, 0xab, 0x21, 0x1b, 0xc4, 0x30, 0x69, 0xee,
	0x7f, 0x09, 0xe6, 0x6b, 0x99, 0x98, 0x34, 0x07,
	0x7b, 0x9a, 0x58, 0xb2, 0xe8, 0x77, 0xe0, 0xba,
	0x7d, 0xab, 0x65, 0xf8, 0xba, 0x2a, 0xcb, 0x9a,
	0x00, 0x91, 0x00, 0x03, 0x04, 0x41, 0x63, 0x78,
	0x35, 0x63, 0x81, 0x59, 0x02, 0xb3, 0x30, 0x82,
	0x02, 0xaf, 0x30, 0x82, 0x01, 0x97, 0xa0, 0x03,
	0x02, 0x01, 0x02, 0x02, 0x04, 0x48, 0x5b, 0x3d,
	0xb6, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
	0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00,
	0x30, 0x21, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03,
	0x55, 0x04, 0x03, 0x0c, 0x16, 0x59, 0x75, 0x62,
	0x00, 0x91, 0x00, 0x03, 0x05, 0x69, 0x63, 0x6f,
	0x20, 0x46, 0x49, 0x44, 0x4f, 0x20, 0x50, 0x72,
	0x65, 0x76, 0x69, 0x65, 0x77, 0x20, 0x43, 0x41,
	0x30, 0x1e, 0x17, 0x0d, 0x31, 0x38, 0x30, 0x34,
	0x31, 0x32, 0x31, 0x30, 0x35, 0x37, 0x31, 0x30,
	0x5a, 0x17, 0x0d, 0x31, 0x38, 0x31, 0x32, 0x33,
	0x31, 0x31, 0x30, 0x35, 0x37, 0x31, 0x30, 0x5a,
	0x30, 0x6f, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03,
	0x00, 0x91, 0x00, 0x03, 0x06, 0x55, 0x04, 0x06,
	0x13, 0x02, 0x53, 0x45, 0x31, 0x12, 0x30, 0x10,
	0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x09, 0x59,
	0x75, 0x62, 0x69, 0x63, 0x6f, 0x20, 0x41, 0x42,
	0x31, 0x22, 0x30, 0x20, 0x06, 0x03, 0x55, 0x04,
	0x0b, 0x0c, 0x19, 0x41, 0x75, 0x74, 0x68, 0x65,
	0x6e, 0x74, 0x69, 0x63, 0x61, 0x74, 0x6f, 0x72,
	0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61,
	0x00, 0x91, 0x00, 0x03, 0x07, 0x74, 0x69, 0x6f,
	0x6e, 0x31, 0x28, 0x30, 0x26, 0x06, 0x03, 0x55,
	0x04, 0x03, 0x0c, 0x1f, 0x59, 0x75, 0x62, 0x69,
	0x63, 0x6f, 0x20, 0x55, 0x32, 0x46, 0x20, 0x45,
	0x45, 0x20, 0x53, 0x65, 0x72, 0x69, 0x61, 0x6c,
	0x20, 0x31, 0x32, 0x31, 0x33, 0x39, 0x33, 0x39,
	0x31, 0x32, 0x36, 0x30, 0x59, 0x30, 0x13, 0x06,
	0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
	0x00, 0x91, 0x00, 0x03, 0x08, 0x06, 0x08, 0x2a,
	0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
	0x42, 0x00, 0x04, 0xfb, 0x2c, 0xdd, 0x30, 0x43,
	0x28, 0xc5, 0x72, 0x4a, 0x50, 0xcc, 0xe6, 0xf6,
	0x0b, 0xad, 0x7d, 0x27, 0xa9, 0x1b, 0x59, 0xe1,
	0xe6, 0x6f, 0x29, 0x7b, 0x89, 0xc9, 0xd4, 0x3d,
	0xc2, 0xb2, 0xc7, 0x78, 0x89, 0xb4, 0xf0, 0xff,
	0x9d, 0x02, 0x28, 0xcb, 0x94, 0x6d, 0xfc, 0xe0,
	0x00, 0x91, 0x00, 0x03, 0x09, 0x1b, 0x19, 0x58,
	0x9b, 0x67, 0x80, 0x4a, 0xac, 0x97, 0x7f, 0x28,
	0x18, 0x9c, 0xcd, 0xb3, 0x25, 0x74, 0xca, 0x28,
	0xa3, 0x6c, 0x30, 0x6a, 0x30, 0x22, 0x06, 0x09,
	0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xc4, 0x0a,
	0x02, 0x04, 0x15, 0x31, 0x2e, 0x33, 0x2e, 0x36,
	0x2e, 0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34,
	0x31, 0x34, 0x38, 0x32, 0x2e, 0x31, 0x2e, 0x36,
	0x00, 0x91, 0x00, 0x03, 0x0a, 0x30, 0x13, 0x06,
	0x0b, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xe5,
	0x1c, 0x02, 0x01, 0x01, 0x04, 0x04, 0x03, 0x02,
	0x04, 0x30, 0x30, 0x21, 0x06, 0x0b, 0x2b, 0x06,
	0x01, 0x04, 0x01, 0x82, 0xe5, 0x1c, 0x01, 0x01,
	0x04, 0x04, 0x12, 0x04, 0x10, 0xf8, 0xa0, 0x11,
	0xf3, 0x8c, 0x0a, 0x4d, 0x15, 0x80, 0x06, 0x17,
	0x11, 0x1f, 0x9e, 0xdc, 0x7d, 0x30, 0x0c, 0x06,
	0x00, 0x91, 0x00, 0x03, 0x0b, 0x03, 0x55, 0x1d,
	0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00,
	0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
	0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03,
	0x82, 0x01, 0x01, 0x00, 0x32, 0xf3, 0xe4, 0xbd,
	0x58, 0xd7, 0x42, 0x2b, 0xaf, 0x49, 0x99, 0x86,
	0x08, 0x1f, 0x0d, 0xa9, 0x3b, 0xc6, 0xaa, 0x1c,
	0x72, 0x11, 0xf9, 0x28, 0x53, 0xeb, 0xf3, 0xeb,
	0x00, 0x91, 0x00, 0x03, 0x0c, 0x73, 0xda, 0x69,
	0x3b, 0x06, 0xde, 0x31, 0x33, 0x8e, 0x5d, 0x02,
	0xec, 0xf6, 0x76, 0xe9, 0x5c, 0x42, 0xbe, 0xa5,
	0x8f, 0x25, 0xd3, 0x37, 0x3f, 0x77, 0xbb, 0x2a,
	0x9d, 0x7c, 0xb2, 0x3e, 0x11, 0x8c, 0x41, 0xd4,
	0x9a, 0x4c, 0x9a, 0xd8, 0xf3, 0xe2, 0xa4, 0xec,
	0x01, 0x77, 0x7a, 0x74, 0xa8, 0xc4, 0x12, 0x43,
	0xc3, 0x1e, 0xce, 0x20, 0x8f, 0x2d, 0x0f, 0x6e,
	0x00, 0x91, 0x00, 0x03, 0x0d, 0xbc, 0x61, 0x9b,
	0xe1, 0x84, 0xa1, 0x72, 0xf6, 0xa9, 0xac, 0xcb,
	0xf8, 0x73, 0x6d, 0x5b, 0xe2, 0x98, 0xb3, 0x6b,
	0xec, 0xe7, 0x1e, 0x77, 0x8d, 0x0a, 0x69, 0xaa,
	0xf9, 0x94, 0xb8, 0x63, 0x6d, 0xe8, 0xfa, 0xf6,
	0x2f, 0xd3, 0xce, 0x7f, 0x04, 0x4c, 0x32, 0x2c,
	0xf7, 0x26, 0x3e, 0x34, 0x99, 0xe6, 0xa5, 0xb2,
	0xb0, 0x2a, 0xbb, 0xad, 0x5b, 0xd9, 0xec, 0xe5,
	0x00, 0x91, 0x00, 0x03, 0x0e, 0xb0, 0x71, 0x4d,
	0x73, 0xbb, 0x94, 0x61, 0x49, 0x9c, 0x94, 0x2a,
	0x5f, 0x1d, 0xcc, 0xaf, 0x65, 0x03, 0x3b, 0x39,
	0x39, 0xd4, 0x47, 0xd9, 0xfc, 0xc4, 0x7b, 0x0b,
	0x16, 0xd8, 0xe9, 0x01, 0xfc, 0xec, 0x3f, 0x8c,
	0x1b, 0xc0, 0xc6, 0xac, 0x0b, 0x5d, 0x74, 0xc7,
	0xbb, 0x03, 0x05, 0x69, 0x17, 0xe9, 0x98, 0x1a,
	0x19, 0xb9, 0x09, 0x5c, 0xa1, 0xf4, 0xab, 0x9f,
	0x00, 0x91, 0x00, 0x03, 0x0f, 0x02, 0x7c, 0x28,
	0x0f, 0x8a, 0xf9, 0xed, 0x1d, 0x29, 0x3c, 0xf6,
	0xcc, 0x2f, 0x04, 0x6d, 0x9a, 0xd6, 0x62, 0xb4,
	0xa9, 0x6e, 0xb1, 0xca, 0xca, 0xac, 0x5e, 0x05,
	0x3e, 0x83, 0x91, 0x47, 0x7c, 0x1f, 0x8b, 0x60,
	0x01, 0xde, 0x65, 0x3a, 0xbf, 0xf2, 0xaa, 0xbb,
	0x55, 0x98, 0x86, 0x91, 0x7e, 0xad, 0x3b, 0x36,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

/*
 * Collection of HID reports from an authenticator issued with a U2F
 * registration using the example parameters above.
 */
static const uint8_t dummy_wire_data_u2f[] = {
	0xff, 0xff, 0xff, 0xff, 0x86, 0x00, 0x11, 0x8e,
	0x80, 0xd0, 0xe2, 0x3b, 0x24, 0x93, 0xea, 0x00,
	0x00, 0x99, 0x01, 0x02, 0x01, 0x01, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x00, 0x02, 0x69,
	0x85, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x83, 0x03, 0x1e, 0x05,
	0x04, 0x9f, 0xa0, 0xf9, 0x0d, 0x4c, 0xf4, 0xae,
	0x96, 0x3c, 0xb7, 0x46, 0xb7, 0x5c, 0x9d, 0x8b,
	0x48, 0x19, 0xdf, 0xc4, 0xad, 0xea, 0xb2, 0x70,
	0x58, 0x72, 0xd9, 0xce, 0x75, 0xf5, 0xe6, 0x8e,
	0x0f, 0x9c, 0x0e, 0x2e, 0x62, 0x3e, 0x91, 0xd3,
	0x7b, 0x97, 0x46, 0x60, 0xb9, 0x57, 0x13, 0x97,
	0x26, 0xae, 0x0f, 0xb3, 0x8f, 0x2e, 0x9b, 0x3f,
	0x00, 0x00, 0x99, 0x01, 0x00, 0xa5, 0x55, 0xec,
	0x8c, 0x25, 0x7c, 0x65, 0xb7, 0x09, 0x40, 0x48,
	0xae, 0xa8, 0xcb, 0xa1, 0x91, 0xac, 0x40, 0x24,
	0xf2, 0x34, 0x6e, 0x3a, 0x8f, 0xa5, 0xb7, 0x48,
	0x54, 0x6e, 0xfb, 0xf4, 0x37, 0x88, 0x69, 0x79,
	0x6f, 0x12, 0xc1, 0x32, 0xdf, 0x15, 0x5d, 0x6e,
	0x82, 0x54, 0xc0, 0x6e, 0x56, 0x4f, 0x3a, 0x9c,
	0xc3, 0x96, 0x7a, 0xde, 0xa5, 0xfe, 0xec, 0xd1,
	0x00, 0x00, 0x99, 0x01, 0x01, 0x5a, 0x21, 0x85,
	0x0e, 0x25, 0x7b, 0x8d, 0x6e, 0x1d, 0x32, 0x29,
	0xdb, 0x21, 0xb0, 0xa3, 0x30, 0x82, 0x02, 0x4f,
	0x30, 0x82, 0x01, 0x37, 0xa0, 0x03, 0x02, 0x01,
	0x02, 0x02, 0x04, 0x2a, 0xd9, 0x6a, 0xf3, 0x30,
	0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x2e,
	0x31, 0x2c, 0x30, 0x2a, 0x06, 0x03, 0x55, 0x04,
	0x00, 0x00, 0x99, 0x01, 0x02, 0x03, 0x13, 0x23,
	0x59, 0x75, 0x62, 0x69, 0x63, 0x6f, 0x20, 0x55,
	0x32, 0x46, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20,
	0x43, 0x41, 0x20, 0x53, 0x65, 0x72, 0x69, 0x61,
	0x6c, 0x20, 0x34, 0x35, 0x37, 0x32, 0x30, 0x30,
	0x36, 0x33, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x31,
	0x34, 0x30, 0x38, 0x30, 0x31, 0x30, 0x30, 0x30,
	0x30, 0x30, 0x30, 0x5a, 0x18, 0x0f, 0x32, 0x30,
	0x00, 0x00, 0x99, 0x01, 0x03, 0x35, 0x30, 0x30,
	0x39, 0x30, 0x34, 0x30, 0x30, 0x30, 0x30, 0x30,
	0x30, 0x5a, 0x30, 0x31, 0x31, 0x2f, 0x30, 0x2d,
	0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x26, 0x59,
	0x75, 0x62, 0x69, 0x63, 0x6f, 0x20, 0x55, 0x32,
	0x46, 0x20, 0x45, 0x45, 0x20, 0x53, 0x65, 0x72,
	0x69, 0x61, 0x6c, 0x20, 0x32, 0x33, 0x39, 0x32,
	0x35, 0x37, 0x33, 0x34, 0x35, 0x31, 0x36, 0x35,
	0x00, 0x00, 0x99, 0x01, 0x04, 0x35, 0x30, 0x33,
	0x38, 0x37, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07,
	0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
	0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01,
	0x07, 0x03, 0x42, 0x00, 0x04, 0x2f, 0xe1, 0xa2,
	0x3e, 0xbf, 0xa5, 0x5b, 0x3e, 0x46, 0x1d, 0x59,
	0xa4, 0x35, 0x22, 0xd7, 0x97, 0x48, 0x98, 0x1c,
	0xba, 0x6d, 0x28, 0x9a, 0x98, 0xf1, 0xbd, 0x7d,
	0x00, 0x00, 0x99, 0x01, 0x05, 0xff, 0x65, 0x66,
	0x80, 0xdb, 0xbb, 0xed, 0xbc, 0x2b, 0xae, 0x60,
	0x7e, 0x6e, 0xf7, 0x72, 0xf5, 0x76, 0xb0, 0x4d,
	0x54, 0xc4, 0xe5, 0xf3, 0x2f, 0x59, 0x6f, 0x26,
	0xe6, 0x11, 0x15, 0xc7, 0x27, 0x2c, 0xf6, 0xca,
	0x75, 0x94, 0xa3, 0x3b, 0x30, 0x39, 0x30, 0x22,
	0x06, 0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82,
	0xc4, 0x0a, 0x02, 0x04, 0x15, 0x31, 0x2e, 0x33,
	0x00, 0x00, 0x99, 0x01, 0x06, 0x2e, 0x36, 0x2e,
	0x31, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x34, 0x31,
	0x34, 0x38, 0x32, 0x2e, 0x31, 0x2e, 0x32, 0x30,
	0x13, 0x06, 0x0b, 0x2b, 0x06, 0x01, 0x04, 0x01,
	0x82, 0xe5, 0x1c, 0x02, 0x01, 0x01, 0x04, 0x04,
	0x03, 0x02, 0x04, 0x30, 0x30, 0x0d, 0x06, 0x09,
	0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
	0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00,
	0x00, 0x00, 0x99, 0x01, 0x07, 0x85, 0x6a, 0xfa,
	0x8b, 0xcf, 0x4f, 0x3f, 0x62, 0x5f, 0x29, 0x1b,
	0xc1, 0x15, 0x8e, 0x3c, 0x7e, 0xbd, 0x25, 0x52,
	0xbc, 0xf7, 0x57, 0x07, 0x53, 0xf5, 0x12, 0x1d,
	0xa6, 0xa5, 0x4d, 0x24, 0xcc, 0xcf, 0xae, 0x27,
	0xce, 0xd6, 0xab, 0x31, 0x12, 0x8c, 0x29, 0x7e,
	0x5b, 0x5b, 0x89, 0x05, 0xdd, 0xa0, 0x20, 0x17,
	0x93, 0x1f, 0x1f, 0x5f, 0x59, 0x25, 0x93, 0x59,
	0x00, 0x00, 0x99, 0x01, 0x08, 0x51, 0xfc, 0x00,
	0x4b, 0xcb, 0xe2, 0x0a, 0xdd, 0x7d, 0x8d, 0x05,
	0x2f, 0x95, 0x43, 0xb3, 0x49, 0x6c, 0x15, 0xb8,
	0x31, 0x0e, 0x10, 0xcb, 0xd9, 0xbb, 0x05, 0x38,
	0x27, 0x4f, 0x58, 0x3e, 0xad, 0x1f, 0x45, 0x12,
	0x88, 0xc3, 0xea, 0x76, 0xd0, 0x70, 0xad, 0x44,
	0xe5, 0x3a, 0xfe, 0xa8, 0xf2, 0x2d, 0x1f, 0x73,
	0x62, 0x5f, 0xf2, 0xd5, 0x89, 0xfe, 0x30, 0xdf,
	0x00, 0x00, 0x99, 0x01, 0x09, 0x26, 0x62, 0xcb,
	0x7c, 0xbb, 0x7c, 0x99, 0x61, 0x80, 0xad, 0xcf,
	0xa9, 0x8a, 0x4d, 0x01, 0x2c, 0xf3, 0x13, 0x46,
	0xcd, 0x11, 0x74, 0x6a, 0x58, 0x48, 0xe8, 0xbe,
	0xed, 0xf3, 0xe3, 0x0c, 0xcb, 0xd9, 0xc1, 0xdd,
	0x22, 0x16, 0x71, 0xb2, 0x83, 0x88, 0x61, 0xf6,
	0x5a, 0x45, 0x36, 0x23, 0xb5, 0x18, 0xd5, 0x56,
	0x7f, 0xa8, 0xf0, 0xa3, 0xce, 0x10, 0x5d, 0xf4,
	0x00, 0x00, 0x99, 0x01, 0x0a, 0xf1, 0x39, 0x53,
	0xe1, 0x14, 0xea, 0x59, 0xe0, 0xa7, 0xf2, 0xfe,
	0x66, 0x88, 0x67, 0x43, 0x2e, 0x52, 0xfd, 0x6a,
	0x2f, 0x64, 0xf7, 0x3c, 0x48, 0xcd, 0x9b, 0x38,
	0xf2, 0xdf, 0xba, 0x2c, 0x7a, 0x4b, 0x3b, 0x11,
	0x28, 0xdf, 0x26, 0xd6, 0x6a, 0x24, 0xf8, 0x95,
	0xdd, 0xa0, 0xb6, 0x11, 0x80, 0xf4, 0x14, 0x4f,
	0x6b, 0x70, 0x75, 0xc3, 0x18, 0xa4, 0x9a, 0xe0,
	0x00, 0x00, 0x99, 0x01, 0x0b, 0x8b, 0x58, 0xd3,
	0x6a, 0xdb, 0x1e, 0x30, 0x53, 0x67, 0x2b, 0x17,
	0xc5, 0xa1, 0x9f, 0x7f, 0x0a, 0x22, 0xf1, 0x0e,
	0x94, 0x30, 0x44, 0x02, 0x20, 0x07, 0x5c, 0x4f,
	0xd2, 0x83, 0xb6, 0x9f, 0x0a, 0x4a, 0x4d, 0x4b,
	0x08, 0x35, 0xeb, 0xc0, 0x7e, 0x4a, 0x14, 0x2e,
	0xc7, 0x8c, 0xd6, 0x64, 0x2f, 0xd3, 0x1e, 0xcc,
	0xb5, 0xe8, 0x42, 0xea, 0xf6, 0x02, 0x20, 0x6b,
	0x00, 0x00, 0x99, 0x01, 0x0c, 0x5a, 0xba, 0x4a,
	0xc8, 0xd7, 0x89, 0xcc, 0x77, 0xe6, 0xb9, 0xa3,
	0x34, 0xea, 0x06, 0x85, 0x72, 0xc6, 0x28, 0xa8,
	0x7a, 0xaa, 0x19, 0x88, 0x34, 0xbb, 0xdc, 0x64,
	0x90, 0x0a, 0xdb, 0x39, 0x90, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

int    LLVMFuzzerTestOneInput(const uint8_t *, size_t);
size_t LLVMFuzzerCustomMutator(uint8_t *, size_t, size_t, unsigned int);

static int
unpack(const uint8_t *ptr, size_t len, struct param *p) NO_MSAN
{
	uint8_t **pp = (void *)&ptr;

	if (unpack_byte(TAG_RK, pp, &len, &p->rk) < 0 ||
	    unpack_byte(TAG_TYPE, pp, &len, &p->type) < 0 ||
	    unpack_byte(TAG_U2F, pp, &len, &p->u2f) < 0 ||
	    unpack_byte(TAG_UV, pp, &len, &p->uv) < 0 ||
	    unpack_byte(TAG_EXCL_COUNT, pp, &len, &p->excl_count) < 0 ||
	    unpack_string(TAG_PIN, pp, &len, p->pin) < 0 ||
	    unpack_string(TAG_RP_ID, pp, &len, p->rp_id) < 0 ||
	    unpack_string(TAG_RP_NAME, pp, &len, p->rp_name) < 0 ||
	    unpack_string(TAG_USER_ICON, pp, &len, p->user_icon) < 0 ||
	    unpack_string(TAG_USER_NAME, pp, &len, p->user_name) < 0 ||
	    unpack_string(TAG_USER_NICK, pp, &len, p->user_nick) < 0 ||
	    unpack_int(TAG_EXT, pp, &len, &p->ext) < 0 ||
	    unpack_int(TAG_SEED, pp, &len, &p->seed) < 0 ||
	    unpack_blob(TAG_CDH, pp, &len, &p->cdh) < 0 ||
	    unpack_blob(TAG_USER_ID, pp, &len, &p->user_id) < 0 ||
	    unpack_blob(TAG_WIRE_DATA, pp, &len, &p->wire_data) < 0 ||
	    unpack_blob(TAG_EXCL_CRED, pp, &len, &p->excl_cred) < 0)
		return (-1);

	return (0);
}

static size_t
pack(uint8_t *ptr, size_t len, const struct param *p)
{
	const size_t max = len;

	if (pack_byte(TAG_RK, &ptr, &len, p->rk) < 0 ||
	    pack_byte(TAG_TYPE, &ptr, &len, p->type) < 0 ||
	    pack_byte(TAG_U2F, &ptr, &len, p->u2f) < 0 ||
	    pack_byte(TAG_UV, &ptr, &len, p->uv) < 0 ||
	    pack_byte(TAG_EXCL_COUNT, &ptr, &len, p->excl_count) < 0 ||
	    pack_string(TAG_PIN, &ptr, &len, p->pin) < 0 ||
	    pack_string(TAG_RP_ID, &ptr, &len, p->rp_id) < 0 ||
	    pack_string(TAG_RP_NAME, &ptr, &len, p->rp_name) < 0 ||
	    pack_string(TAG_USER_ICON, &ptr, &len, p->user_icon) < 0 ||
	    pack_string(TAG_USER_NAME, &ptr, &len, p->user_name) < 0 ||
	    pack_string(TAG_USER_NICK, &ptr, &len, p->user_nick) < 0 ||
	    pack_int(TAG_EXT, &ptr, &len, p->ext) < 0 ||
	    pack_int(TAG_SEED, &ptr, &len, p->seed) < 0 ||
	    pack_blob(TAG_CDH, &ptr, &len, &p->cdh) < 0 ||
	    pack_blob(TAG_USER_ID, &ptr, &len, &p->user_id) < 0 ||
	    pack_blob(TAG_WIRE_DATA, &ptr, &len, &p->wire_data) < 0 ||
	    pack_blob(TAG_EXCL_CRED, &ptr, &len, &p->excl_cred) < 0)
		return (0);

	return (max - len);
}

static void
make_cred(fido_cred_t *cred, uint8_t u2f, int type, const struct blob *cdh,
    const char *rp_id, const char *rp_name, struct blob *user_id,
    const char *user_name, const char *user_nick, const char *user_icon,
    int ext, uint8_t rk, uint8_t uv, const char *pin, uint8_t excl_count,
    struct blob *excl_cred)
{
	fido_dev_t	*dev;
	fido_dev_io_t	 io;

	io.open = dev_open;
	io.close = dev_close;
	io.read = dev_read;
	io.write = dev_write;

	if ((dev = fido_dev_new()) == NULL || fido_dev_set_io_functions(dev,
	    &io) != FIDO_OK || fido_dev_open(dev, "nodev") != FIDO_OK) {
		fido_dev_free(&dev);
		return;
	}

	if (u2f & 1)
		fido_dev_force_u2f(dev);

	for (uint8_t i = 0; i < excl_count; i++)
		fido_cred_exclude(cred, excl_cred->body, excl_cred->len);

	fido_cred_set_type(cred, type);
	fido_cred_set_clientdata_hash(cred, cdh->body, cdh->len);
	fido_cred_set_rp(cred, rp_id, rp_name);
	fido_cred_set_user(cred, user_id->body, user_id->len, user_name,
	    user_nick, user_icon);
	fido_cred_set_extensions(cred, ext);
	if (rk & 1)
		fido_cred_set_rk(cred, FIDO_OPT_TRUE);
	if (uv & 1)
		fido_cred_set_uv(cred, FIDO_OPT_TRUE);

	fido_dev_make_cred(dev, cred, u2f & 1 ? NULL : pin);

	fido_dev_cancel(dev);
	fido_dev_close(dev);
	fido_dev_free(&dev);
}

static void
verify_cred(int type, const unsigned char *cdh_ptr, size_t cdh_len,
    const char *rp_id, const char *rp_name, const unsigned char *authdata_ptr,
    size_t authdata_len, int ext, uint8_t rk, uint8_t uv,
    const unsigned char *x5c_ptr, size_t x5c_len, const unsigned char *sig_ptr,
    size_t sig_len, const char *fmt)
{
	fido_cred_t	*cred;
	uint8_t		 flags;

	if ((cred = fido_cred_new()) == NULL) {
		warnx("%s: fido_cred_new", __func__);
		return;
	}

	fido_cred_set_type(cred, type);
	fido_cred_set_clientdata_hash(cred, cdh_ptr, cdh_len);
	fido_cred_set_rp(cred, rp_id, rp_name);
	if (fido_cred_set_authdata(cred, authdata_ptr, authdata_len) != FIDO_OK)
		fido_cred_set_authdata_raw(cred, authdata_ptr, authdata_len);
	fido_cred_set_extensions(cred, ext);
	fido_cred_set_x509(cred, x5c_ptr, x5c_len);
	fido_cred_set_sig(cred, sig_ptr, sig_len);

	if (rk & 1)
		fido_cred_set_rk(cred, FIDO_OPT_TRUE);
	if (uv & 1)
		fido_cred_set_uv(cred, FIDO_OPT_TRUE);
	if (fmt)
		fido_cred_set_fmt(cred, fmt);

	fido_cred_verify(cred);
	fido_cred_verify_self(cred);

	consume(fido_cred_pubkey_ptr(cred), fido_cred_pubkey_len(cred));
	consume(fido_cred_id_ptr(cred), fido_cred_id_len(cred));
	consume(fido_cred_user_id_ptr(cred), fido_cred_user_id_len(cred));
	consume(fido_cred_user_name(cred), xstrlen(fido_cred_user_name(cred)));
	consume(fido_cred_display_name(cred),
	    xstrlen(fido_cred_display_name(cred)));

	flags = fido_cred_flags(cred);
	consume(&flags, sizeof(flags));
	type = fido_cred_type(cred);
	consume(&type, sizeof(type));

	fido_cred_free(&cred);
}

int
LLVMFuzzerTestOneInput(const uint8_t *data, size_t size)
{
	struct param	 p;
	fido_cred_t	*cred = NULL;
	int		 cose_alg = 0;

	memset(&p, 0, sizeof(p));

	if (unpack(data, size, &p) < 0)
		return (0);

	srandom((unsigned int)p.seed);

	fido_init(0);

	if ((cred = fido_cred_new()) == NULL)
		return (0);

	set_wire_data(p.wire_data.body, p.wire_data.len);

	switch (p.type & 3) {
	case 0:
		cose_alg = COSE_ES256;
		break;
	case 1:
		cose_alg = COSE_RS256;
		break;
	default:
		cose_alg = COSE_EDDSA;
		break;
	}

	make_cred(cred, p.u2f, cose_alg, &p.cdh, p.rp_id, p.rp_name,
	    &p.user_id, p.user_name, p.user_nick, p.user_icon, p.ext, p.rk,
	    p.uv, p.pin, p.excl_count, &p.excl_cred);

	verify_cred(cose_alg,
	    fido_cred_clientdata_hash_ptr(cred),
	    fido_cred_clientdata_hash_len(cred), fido_cred_rp_id(cred),
	    fido_cred_rp_name(cred), fido_cred_authdata_ptr(cred),
	    fido_cred_authdata_len(cred), p.ext, p.rk, p.uv,
	    fido_cred_x5c_ptr(cred), fido_cred_x5c_len(cred),
	    fido_cred_sig_ptr(cred), fido_cred_sig_len(cred),
	    fido_cred_fmt(cred));

	fido_cred_free(&cred);

	return (0);
}

static size_t
pack_dummy(uint8_t *ptr, size_t len)
{
	struct param	dummy;
	uint8_t		blob[16384];
	size_t		blob_len;

	memset(&dummy, 0, sizeof(dummy));

	dummy.type = 1;
	dummy.ext = FIDO_EXT_HMAC_SECRET;

	strlcpy(dummy.pin, dummy_pin, sizeof(dummy.pin));
	strlcpy(dummy.rp_id, dummy_rp_id, sizeof(dummy.rp_id));
	strlcpy(dummy.rp_name, dummy_rp_name, sizeof(dummy.rp_name));
	strlcpy(dummy.user_icon, dummy_user_icon, sizeof(dummy.user_icon));
	strlcpy(dummy.user_name, dummy_user_name, sizeof(dummy.user_name));
	strlcpy(dummy.user_nick, dummy_user_nick, sizeof(dummy.user_nick));

	dummy.cdh.len = sizeof(dummy_cdh);
	dummy.user_id.len = sizeof(dummy_user_id);
	dummy.wire_data.len = sizeof(dummy_wire_data_fido);

	memcpy(&dummy.cdh.body, &dummy_cdh, dummy.cdh.len);
	memcpy(&dummy.user_id.body, &dummy_user_id, dummy.user_id.len);
	memcpy(&dummy.wire_data.body, &dummy_wire_data_fido,
	    dummy.wire_data.len);

	blob_len = pack(blob, sizeof(blob), &dummy);
	assert(blob_len != 0);

	if (blob_len > len) {
		memcpy(ptr, blob, len);
		return (len);
	}

	memcpy(ptr, blob, blob_len);

	return (blob_len);
}

size_t
LLVMFuzzerCustomMutator(uint8_t *data, size_t size, size_t maxsize,
    unsigned int seed) NO_MSAN
{
	struct param	p;
	uint8_t		blob[16384];
	size_t		blob_len;

	memset(&p, 0, sizeof(p));

	if (unpack(data, size, &p) < 0)
		return (pack_dummy(data, maxsize));

	mutate_byte(&p.rk);
	mutate_byte(&p.type);
	mutate_byte(&p.u2f);
	mutate_byte(&p.uv);
	mutate_byte(&p.excl_count);

	mutate_int(&p.ext);
	p.seed = (int)seed;

	mutate_blob(&p.cdh);
	mutate_blob(&p.user_id);

	if (p.u2f & 1) {
		p.wire_data.len = sizeof(dummy_wire_data_u2f);
		memcpy(&p.wire_data.body, &dummy_wire_data_u2f,
		    p.wire_data.len);
	} else {
		p.wire_data.len = sizeof(dummy_wire_data_fido);
		memcpy(&p.wire_data.body, &dummy_wire_data_fido,
		    p.wire_data.len);
	}

	mutate_blob(&p.wire_data);
	mutate_blob(&p.excl_cred);

	mutate_string(p.pin);
	mutate_string(p.user_icon);
	mutate_string(p.user_name);
	mutate_string(p.user_nick);
	mutate_string(p.rp_id);
	mutate_string(p.rp_name);

	blob_len = pack(blob, sizeof(blob), &p);

	if (blob_len == 0 || blob_len > maxsize)
		return (0);

	memcpy(data, blob, blob_len);

	return (blob_len);
}
