// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "services/ui/ws/frame_generator.h"

#include <memory>

#include "base/memory/ptr_util.h"
#include "base/test/test_message_loop.h"
#include "cc/quads/render_pass.h"
#include "cc/quads/shared_quad_state.h"
#include "services/ui/ws/ids.h"
#include "services/ui/ws/platform_display_init_params.h"
#include "services/ui/ws/server_window.h"
#include "services/ui/ws/server_window_compositor_frame_sink_manager.h"
#include "services/ui/ws/test_server_window_delegate.h"
#include "services/ui/ws/test_utils.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace ui {
namespace ws {
namespace test {
namespace {

// Typical id for the display root ServerWindow.
constexpr WindowId kRootDisplayId(0, 2);
const base::UnguessableToken kArbitraryToken = base::UnguessableToken::Create();

// Makes the window visible and creates the default surface for it.
void InitWindow(ServerWindow* window) {
  window->SetVisible(true);
  ServerWindowCompositorFrameSinkManager* compositor_frame_sink_manager =
      window->GetOrCreateCompositorFrameSinkManager();
  compositor_frame_sink_manager->SetLatestSurfaceInfo(cc::SurfaceInfo(
      cc::SurfaceId(cc::FrameSinkId(WindowIdToTransportId(window->id()), 0),
                    cc::LocalFrameId(1u, kArbitraryToken)),
      1.0f, gfx::Size(100, 100)));
}

}  // namespace

class FrameGeneratorTest : public testing::Test {
 public:
  FrameGeneratorTest()
      : root_window_(base::MakeUnique<ServerWindow>(&window_delegate_,
                                                    kRootDisplayId)) {}
  ~FrameGeneratorTest() override {}

  // Calls DrawWindow() on |frame_generator_|
  void DrawWindow(cc::RenderPass* pass);

  ServerWindow* root_window() { return root_window_.get(); }

  TestServerWindowDelegate* test_window_delegate() { return &window_delegate_; }

 private:
  // testing::Test:
  void SetUp() override;
  void TearDown() override;

  std::unique_ptr<FrameGenerator> frame_generator_;
  std::unique_ptr<TestFrameGeneratorDelegate> frame_generator_delegate_;
  TestServerWindowDelegate window_delegate_;
  std::unique_ptr<ServerWindow> root_window_;

  // Needed so that Mojo classes can be initialized for ServerWindow use.
  base::TestMessageLoop message_loop_;

  DISALLOW_COPY_AND_ASSIGN(FrameGeneratorTest);
};

void FrameGeneratorTest::DrawWindow(cc::RenderPass* pass) {
  frame_generator_->DrawWindow(pass, root_window_.get());
}

void FrameGeneratorTest::SetUp() {
  testing::Test::SetUp();
  frame_generator_delegate_ =
      base::MakeUnique<TestFrameGeneratorDelegate>(root_window_.get());
  PlatformDisplayInitParams init_params;
  frame_generator_ = base::MakeUnique<FrameGenerator>(
      frame_generator_delegate_.get(), root_window_.get());
  frame_generator_->set_device_scale_factor(
      init_params.metrics.device_scale_factor);
  InitWindow(root_window());
}

void FrameGeneratorTest::TearDown() {
  frame_generator_.reset();
  frame_generator_delegate_.reset();
}

// Tests correctness of the SharedQuadStateList generated by
// FrameGenerator::DrawWindow().
TEST_F(FrameGeneratorTest, DrawWindow) {
  ServerWindow child_window(test_window_delegate(), WindowId(1, 1));
  root_window()->Add(&child_window);
  InitWindow(&child_window);
  const float child_opacity = .4f;
  child_window.SetOpacity(child_opacity);

  std::unique_ptr<cc::RenderPass> render_pass = cc::RenderPass::Create();
  DrawWindow(render_pass.get());
  cc::SharedQuadStateList* quad_state_list =
      &render_pass->shared_quad_state_list;

  EXPECT_EQ(1u, quad_state_list->size());
  cc::SharedQuadState* root_sqs = quad_state_list->back();
  // Opacity should always be 1.
  EXPECT_EQ(1.0f, root_sqs->opacity);
}

}  // namespace test
}  // namespace ws
}  // namespace ui
