// ---------------------------------------------------------------------------
// - Taggable.cpp                                                            -
// - standard object library - taggable class implementation                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Taggable.hpp"
#include "QuarkZone.hpp"

namespace afnix {
  
  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a taggable by name

  Taggable::Taggable (const String& name) {
    d_name = name;
  }

  // create a taggable by name and info

  Taggable::Taggable (const String& name, const String& info) {
    d_name = name;
    d_info = info;
  }
  
  // copy construct this taggable

  Taggable::Taggable (const Taggable& that) {
    that.rdlock ();
    try {
      d_name = that.d_name;
      d_info = that.d_info;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }
  
  // copy move this taggable

  Taggable::Taggable (Taggable&& that) noexcept {
    that.wrlock ();
    try {
      // move base nameable
      Nameable::operator = (static_cast<Nameable&&>(that));
      // copy move locally
      d_name = that.d_name; that.d_name.clear();
      d_info = that.d_info; that.d_info.clear();
    } catch (...) {
      d_name.clear();
      d_info.clear();
    }
    that.unlock ();
  }
  
  // assign a taggable into this one

  Taggable& Taggable::operator = (const Taggable& that) {
    // check for self-assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // assign locally
      d_name = that.d_name;
      d_info = that.d_info;
      // unlock and here
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // move this taggable into this one

  Taggable& Taggable::operator = (Taggable&& that) noexcept {
    // check for self-assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.wrlock ();
    try {
      // move base nameable
      Nameable::operator = (static_cast<Nameable&&>(that));
      // move locally
      d_name = that.d_name; that.d_name.clear();
      d_info = that.d_info; that.d_info.clear();
    } catch (...) {
      d_name.clear();
      d_info.clear();
    }
    unlock ();
    that.unlock ();
    return *this;
  }
  
  // serialize this taggable

  void Taggable::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      d_name.wrstream (os);
      d_info.wrstream (os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this taggable

  void Taggable::rdstream (InputStream& is) {
    wrlock ();
    try {
      d_name.rdstream (is);
      d_info.rdstream (is);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // get the taggable name

  String Taggable::getname (void) const {
    rdlock ();
    try {
      String result = d_name;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the taggable name

  void Taggable::setname (const String& name) {
    wrlock ();
    try {
      d_name = name;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // get the taggable info

  String Taggable::getinfo (void) const {
    rdlock ();
    try {
      String result = d_info;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the taggable info

  void Taggable::setinfo (const String& info) {
    wrlock ();
    try {
      d_info = info;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 3;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SETNAME = zone.intern ("set-name");
  static const long QUARK_SETINFO = zone.intern ("set-info");
  static const long QUARK_GETINFO = zone.intern ("get-info");

  // return true if the given quark is defined

  bool Taggable::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? Nameable::isquark (quark, hflg) : false;
      if (result == false) {
	result = hflg ? Serial::isquark (quark, hflg) : false;
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark

  Object* Taggable::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nullptr) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETINFO) return new String  (getinfo ());
    }
     // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETNAME) {
	String name = argv->getstring (0);
	setname (name);
	return nullptr;
      }
      if (quark == QUARK_SETINFO) {
	String info = argv->getstring (0);
	setinfo (info);
	return nullptr;
      }
    }
    // check the nameable method
    if (Nameable::isquark (quark, true) == true) {
      return Nameable::apply (robj, nset, quark, argv);
    }
    // call the serial method
    return Serial::apply (robj, nset, quark, argv);
  }
}
