package org.eclipse.webdav.internal.kernel;

/*
 * (c) Copyright IBM Corp. 2000, 2001.
 * All Rights Reserved.
 */

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.Writer;

import org.eclipse.webdav.IContext;
import org.eclipse.webdav.IResponse;
import org.eclipse.webdav.internal.kernel.utils.Assert;
import org.w3c.dom.Document;

/**
 * The <code>Response</code> class subclasses <code>Message</code> to
 * add a status field.
 */
public class Response extends Message implements IResponse {

	protected Status status;

	public Response(Status status, IContext context, InputStream body) {
		super();

		this.status = status;
		this.context = context;
		this.body = body;
	}


	public Response(Status status, IContext context, Document body) {
		super();

		this.status = status;
		this.context = context;
		this.body = body;
	}


	public void close() throws IOException {
		if (!hasDocumentBody())
			getInputStream().close();
	}

	public Document getDocumentBody() throws IOException {
		Assert.isTrue(hasDocumentBody(), Policy.bind("error.receiverMissingBody"));

		return (Document) body;
	}

	public InputStream getInputStream() {
		Assert.isTrue(!hasDocumentBody(), Policy.bind("error.receiverHasBody"));

		return (InputStream) body;
	}

	/**
	 * Return the status code for this response.
	 *
	 * <p>Implements the corresponding API in the interface
	 * <code>Response</code>.</p>
	 *
	 * @return the status code
	 * @see Response#getStatusCode()
	 */
	public int getStatusCode() {
		return status.getCode();
	}

	/**
	 * Return the status message for this response.
	 *
	 * <p>Implements the corresponding API in the interface
	 * <code>Response</code>.</p>
	 *
	 * @return the status message
	 * @see Response#getStatusMessage()
	 */
	public String getStatusMessage() {
		return status.getMessage();
	}

	public boolean hasDocumentBody() {
		return (body instanceof Document);
	}

	public String toString() {
		StringBuffer buffer = new StringBuffer();

		buffer.append(status.toHTTPString());
		buffer.append('\n');
		buffer.append(context.toString());
		buffer.append('\n');

		try {
			if (hasDocumentBody()) {
				IDocumentMarshaler marshaler = new DocumentMarshaler();
				Writer writer = new StringWriter();
				marshaler.print(getDocumentBody(), writer, "UTF-8");
				buffer.append(writer.toString());
			} else
				buffer.append(
					"<<"
						+ Policy.bind("label.bytes", String.valueOf(getInputStream().available()))
						+ ">>");
		} catch (IOException exception) {
			buffer.append("<<" + Policy.bind("exception.dumping") + ">>");
		}

		return buffer.toString();
	}
}