#ifndef _RHEO_BASIS_H
#define _RHEO_BASIS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/reference_element.h"
#include "rheolef/point.h"
#include "rheolef/tensor.h"
#include "rheolef/element_constant.h"
#include "rheolef/smart_pointer.h"
namespace rheolef { 

template<class T>
class basis_rep {
public:
    typedef reference_element::size_type      size_type;

    basis_rep (std::string name) : _name(name) {};
    virtual ~basis_rep() {}
    std::string     name()   const { return _name; }
    virtual size_type degree () const = 0;
    virtual size_type size (reference_element hat_K) const = 0;
    virtual void hat_node(
        reference_element                               hat_K,
	std::vector<point_basic<T> >&                   hat_node) const = 0;
    virtual void hat_node(
        reference_element                               hat_K,
	typename std::vector<point_basic<T> >::iterator hat_node) const = 0;
    virtual void eval(
        reference_element                 hat_K,
        const point_basic<T>&             hat_x,
	std::vector<T>&                   values) const = 0;
    virtual void eval(
        reference_element                 hat_K,
        const point_basic<T>&             hat_x,
	typename std::vector<T>::iterator values) const = 0;
    virtual void grad_eval(
        reference_element                               hat_K,
        const point_basic<T>&                           hat_x,
	std::vector<point_basic<T> >&                   values) const = 0;
    virtual void grad_eval(
        reference_element                               hat_K,
        const point_basic<T>&                           hat_x,
	typename std::vector<point_basic<T> >::iterator values) const = 0;
    static basis_rep* make_ptr (std::string name);
protected:
    std::string     _name;
};

/*Class:basis
NAME: @code{basis} - polynomial basis
@cindex  polynomial basis
@clindex basis
@cindex  reference element
@clindex reference_element
SYNOPSYS:
  @noindent
  The @code{basis} class defines functions that evaluates a polynomial
  basis and its derivatives on a point. The polynomial basis
  is designated by a string, e.g. "P0", "P1", "P2", "bubble",...
  indicating the basis. The basis depends also of the reference element:
  triangle, square, tetrahedron (@pxref{reference_element iclass}).
  For instance, on a square, the "P1"
  string designates the common Q1 four-nodes basis on the reference square.

  @noindent
  The nodes associated to the Lagrange polynomial basis
  are also available by its associated accessor.

IMPLEMENTATION NOTE:
  @noindent
  The @code{basis} class 
  is a @pxref{smart_pointer iclass}) class on a @code{basis_rep} class that is a pure virtual base class 
  for effective bases, e.g. basis_P1, basis_P1, etc.
AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   7 january 2004
End:
*/

//<basis:
template<class T>
class basis_basic : public smart_pointer<basis_rep<T> > {
public:

// typedefs:

    typedef basis_rep<T>                           rep;
    typedef smart_pointer<rep>                     base;
    typedef typename basis_rep<T>::size_type       size_type;

// allocators:

    basis_basic (std::string name = "");

// accessors:

    std::string name() const;
    size_type   degree() const;
    size_type   size (reference_element hat_K) const;

    void hat_node(
        reference_element              hat_K,
	std::vector<point_basic<T> >&  hat_node) const;
    void hat_node(
        reference_element                               hat_K,
	typename std::vector<point_basic<T> >::iterator hat_node) const;

    void eval(
        reference_element     hat_K,
        const point_basic<T>& hat_x,
	std::vector<T>&       values) const;
    void eval(
        reference_element                 hat_K,
        const point_basic<T>&             hat_x,
	typename std::vector<T>::iterator values) const;
    
    void grad_eval(
        reference_element    hat_K,
        const point_basic<T>&         hat_x,
	std::vector<point_basic<T> >&  values) const;
    void grad_eval(
        reference_element                               hat_K,
        const point_basic<T>&                           hat_x,
	typename std::vector<point_basic<T> >::iterator values) const;
};
typedef basis_basic<Float> basis;
//>basis:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
template<class T>
inline
basis_basic<T>::basis_basic(std::string name)
 : base (basis_rep<T>::make_ptr(name))
{
}
template<class T>
inline
std::string
basis_basic<T>::name() const
{
    return base::data().name();
}
template<class T>
inline
typename basis_basic<T>::size_type 
basis_basic<T>::degree() const
{
    return base::data().degree();
}
template<class T>
inline
typename basis_basic<T>::size_type
basis_basic<T>::size (reference_element hat_K) const
{
    return base::data().size (hat_K);
}
template<class T>
inline
void
basis_basic<T>::eval(
        reference_element    hat_K,
        const point_basic<T>&         hat_x,
	std::vector<T>&  values) const
{
    return base::data().eval (hat_K, hat_x, values);
}
template<class T>
inline
void
basis_basic<T>::eval(
        reference_element    hat_K,
        const point_basic<T>&         hat_x,
	typename std::vector<T>::iterator values) const
{
    return base::data().eval (hat_K, hat_x, values);
}
template<class T>
inline
void
basis_basic<T>::grad_eval(
        reference_element    hat_K,
        const point_basic<T>&         hat_x,
	std::vector<point_basic<T> >&  values) const
{
    return base::data().grad_eval (hat_K, hat_x, values);
}
template<class T>
inline
void
basis_basic<T>::grad_eval(
        reference_element    hat_K,
        const point_basic<T>&         hat_x,
	typename std::vector<point_basic<T> >::iterator  values) const
{
    return base::data().grad_eval (hat_K, hat_x, values);
}
template<class T>
inline
void
basis_basic<T>::hat_node(
        reference_element    hat_K,
	std::vector<point_basic<T> >&  hat_node) const
{
    return base::data().hat_node (hat_K, hat_node);
}
template<class T>
inline
void
basis_basic<T>::hat_node(
        reference_element    hat_K,
	typename std::vector<point_basic<T> >::iterator  hat_node) const
{
    return base::data().hat_node (hat_K, hat_node);
}

}// namespace rheolef
#endif // _RHEO_BASIS_H
