# Ленивая загрузка путей

При использовании модульного сборщика, результирующая JavaScript-сборка может оказаться довольно большой, что негативно сказывается на времени загрузки страницы. В некоторых случаях было бы эффективнее разделить компоненты каждого пути на отдельные минисборки, и загружать их только при переходе к соответствующему пути.

Совместное использование [асинхронной загрузки компонентов](https://ru.vuejs.org/v2/guide/components.html#Асинхронные-компоненты) Vue и [разделения кода](https://webpack.js.org/guides/code-splitting-async/) Webpack делает реализацию ленивой загрузки компонентов в зависимости от путей тривиальной.

Во-первых, асинхронный компонент можно определить как функцию-фабрику, которая возвращает Promise (который должен разрешиться самим компонентом):

``` js
const Foo = () => Promise.resolve({ /* определение компонента */ })
```

Во-вторых, в Webpack 2 мы можем использовать синтаксис [динамических импортов](https://github.com/tc39/proposal-dynamic-import) для указания точек разделения кода:

``` js
import('./Foo.vue') // возвращает Promise
```

> Примечание: если вы используете Babel, то нужно будет добавить плагин [syntax-dynamic-import](https://babeljs.io/docs/plugins/syntax-dynamic-import/), чтобы Babel смог корректно обработать синтаксис.

Эти два пункта — всё необходимое, чтобы определить асинхронный компонент, который Webpack автоматически вынесет в отдельный чанк:

``` js
const Foo = () => import('./Foo.vue')
```

В конфигурации путей ничего менять не нужно, просто используйте `Foo` как обычно:

``` js
const router = new VueRouter({
  routes: [
    { path: '/foo', component: Foo }
  ]
})
```

### Объединение компонентов в единую минисборку

Иногда может понадобиться объединить в единую минисборку все компоненты, расположенные по определённому пути. Для достижения этой цели можно использовать [именованные минисборки Webpack](https://webpack.js.org/guides/code-splitting-async/#chunk-names), указывая имя используя специальный синтаксис комментариев (в версиях Webpack > 2.4):

``` js
const Foo = () => import(/* webpackChunkName: "group-foo" */ './Foo.vue')
const Bar = () => import(/* webpackChunkName: "group-foo" */ './Bar.vue')
const Baz = () => import(/* webpackChunkName: "group-foo" */ './Baz.vue')
```

Webpack сгруппирует все одноимённые асинхронные модули в единую минисборку.
