#!/bin/sh 
#
### BEGIN INIT INFO
# Provides:          xorp
# Required-Start:    $network $local_fs $remote_fs
# Required-Stop:     $network $local_fs $remote_fs
# Should-Start:      $named
# Should-Stop:       
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: eXtensible Open Router Platform
# Description:       XORP is the eXtensible Open Router Platform. It
#                    implements a number of network protocols such as BGP,
#                    OSPF, RIP/RIPng, IGMP/MLD and PIM-SM.  
### END INIT INFO

PATH=/sbin:/bin:/usr/sbin:/usr/bin
DAEMON=/usr/sbin/xorp_rtrmgr

NAME=xorp
PIDFILE=/var/run/$NAME.pid 
DESC="eXtensible Open Router Platform"

test -x $DAEMON || exit 0
test -x $DAEMON_WRAPPER || exit 0

. /lib/lsb/init-functions

RUN="no"
LOGFILE=/var/log/xorp/router.log
#LOGFILE_ERR=/var/log/xorp/router.err.log
DAEMON_OPTS=""
# maximum sleeping time when waiting for the daemon to die,
# don't set it too low or you might not let xorp die gracefully
MAX_DIETIME=15
# Include xorp defaults if available
if [ -f /etc/default/xorp ] ; then
	. /etc/default/xorp
fi
DAEMON_OPTS="-d -l $LOGFILE -P $PIDFILE $DAEMON_OPTS"

if [ "x$RUN" != "xyes" ] ; then
    log_progress_msg "XORP disabled, please adjust the configuration at /etc/xorp/config.boot "
    log_progress_msg "to your needs and then set RUN to 'yes' in /etc/default/xorp to "
    log_progress_msg "enable the router manager daemon"
    # Return 0, otherwise package installation will fail
    log_end_msg 0
    exit 0
fi

set -e

running_pid() {
# Check if a given process pid's cmdline matches a given name
    pid=$1
    name=$2
    [ -z "$pid" ] && return 1 
    [ ! -d /proc/$pid ] &&  return 1
    cmd=`cat /proc/$pid/cmdline | tr "\000" "\n"|head -n 1 |cut -d : -f 1`
    # Is this the expected child?
    [ "$cmd" != "$name" ] &&  return 1
    return 0
}

running_proc() {
# Check if the process is running looking at /proc
# (works for all users)

    # No pidfile, probably no daemon present
    [ ! -f "$PIDFILE" ] && return 1
    pid=`cat $PIDFILE`

    running_pid $pid $DAEMON || return 1

    return 0
}

running() {
# Check if the process is running
# Use one function or other depending if we are root or not
    running_proc
    return $?
}

start_xorp() {
# Start the process using the wrapper
        start-stop-daemon --start --quiet --pidfile $PIDFILE \
            --name $NAME \
            --exec $DAEMON \
            -- $DAEMON_OPTS
        errcode=$?
	return $errcode
}

stop_xorp() {
# Stop the process using the wrapper
        start-stop-daemon --stop --quiet --pidfile $PIDFILE \
            --exec $DAEMON
        errcode=$?
	return $errcode
}

restart_xorp() {
# Restart the daemon, using the wrapper
# First: only stop if running
        if running ; then
            stop_xorp
# Wait for the service to stop, by increasing the sleeping time
# before we check again
            for wait in `seq $MAX_DIETIME`; do
                 if ! running; then break ; fi
                 sleep $wait
            done
        fi

# If running then we could not get it to stop
        if running ; then
            log_daemon_msg "ERROR: Daemon did not die in the expected time, consider increasing MAX_DIETIME"
            log_end_msg 1
        fi
# Finally: start it again
        start_xorp
	return $?
}

force_stop() {
# Force the process to die killing it manually
	[ ! -e "$PIDFILE" ] && return
	if running ; then
		kill -15 $pid
	# Is it really dead?
		sleep "$MAX_DIETIME"s
		if running ; then
			kill -9 $pid
			sleep "$MAX_DIETIME"s
			if running ; then
				lsb_failure_msg "Cannot kill $NAME (pid=$pid)!"
				exit 1
			fi
		fi
	fi
	rm -f $PIDFILE
        return 0
}


case "$1" in
  start)
	log_daemon_msg "Starting $DESC " "$NAME"
        if running ;  then
            log_progress_msg "apparently already running"
            log_end_msg 0
            exit 0
        fi
        if start_xorp && running ;  then
            log_end_msg 0
        else
            log_end_msg 1
            exit 1
        fi
	;;
  stop)
        log_daemon_msg "Stopping $DESC" "$NAME"
        if running ; then
            stop_xorp
            log_end_msg $?
        else
            log_progress_msg "apparently not running"
            log_end_msg 0
            exit 0
        fi
        ;;
  reload)
        log_warning_msg "Reloading $NAME daemon: not implemented, as the daemon"
        log_warning_msg "cannot re-read the config file (use restart)."
        ;;
  force-stop)
        # First try to stop gracefully
        $0 stop
        if running; then
            log_daemon_msg "Stopping (force) $DESC" "$NAME"
            force_stop
            log_end_msg $?
        fi
	;;
  try-restart)
        log_daemon_msg "Restarting $DESC" "$NAME"
        if running ; then
            restart_xorp
            running
            log_end_msg $?
        else
            log_progress_msg "apparently not running"
            log_end_msg 0
            exit 0
        fi
	;;
  restart|force-reload)
        log_daemon_msg "Restarting $DESC" "$NAME"
        restart_xorp
        running
        log_end_msg $?
	;;
  status)
        log_daemon_msg "Checking status of $DESC" "$NAME"
        if running ;  then
            log_progress_msg "running"
            log_end_msg 0
        else
            log_progress_msg "apparently not running"
            log_end_msg 3
            exit 3
        fi
        ;;
  *)
	N=/etc/init.d/$NAME
	echo "Usage: $N {start|stop|restart|force-reload|status}" >&2
	exit 1
	;;
esac

exit 0
