/*################################################################################
# Linux Management Providers (LMP), Provider Common Library
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr, desmons_frederic@yahoo.fr>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
################################################################################*/

#ifndef _COMMON_H_
#define _COMMON_H_

#ifndef _ERROR_CODES
#define _ERROR_CODES

// Standard return codes
#define OK 0
#define FAILED 1
#define ACCESS_DENIED 2
#define INVALID_NAMESPACE 3
#define INVALID_PARAMETER 4
#define INVALID_CLASS 5
#define NOT_FOUND 6
#define NOT_SUPPORTED 7
#define CLASS_HAS_CHILDREN 8
#define CLASS_HAS_INSTANCES 9
#define INVALID_SUPERCLASS 10
#define ALREADY_EXISTS 11
#define NO_SUCH_PROPERTY 12
#define TYPE_MISMATCH 13
#define QUERY_LANGUAGE_NOT_SUPPORTED 14
#define INVALID_QUERY 15
#define METHOD_NOT_AVAILABLE 16
#define METHOD_NOT_FOUND 17

// Return codes specific to the "cleanup" (unload) method
#define DO_NOT_UNLOAD 50
#define NEVER_UNLOAD 51

// Internal CMPI return codes
#define INVALID_HANDLE 60
#define INVALID_DATA_TYPE 61

// Hosting OS errors
#define ERROR_SYSTEM 100
#define ERROR 200

#endif	//_ERROR_CODES

//Added by Ilsoo Byun (2007-05-09)
//To enhance readability.
#ifndef _COMMON_MACROS
#define _COMMON_MACROS

#define CF_assert(target) if ( int ___rc = (target) != OK  ) return ___rc;
#define CF_assertFail(target) if ( int ___rc = (target) == OK  ) return ___rc
#define CF_assertWithString(target, msg) if ( int ___rc = (target) != OK  ) {errorMessage=(msg); return ___rc;}
#define CF_assertFailWithString(target, msg) if ( int ___rc = (target) == OK  ) {errorMessage=(msg); return ___rc;}
#define CF_assertTrueWithString(target, msg) if (!(target)) {errorMessage=(msg); return FAILED;}
#define CF_assertFalseWithString(target, msg) if ((target)) {errorMessage=(msg); return FAILED;}

#define errmsg(msg) {errorMessage=(msg); return FAILED;}

#define _ARRAY_CMP if (vect1.size() != vect2.size())\
		return false;\
	for (size_t i=0; i < vect1.size(); i++) {\
		if (vect1[i] != vect2[i])\
			return false;\
	}\
	return true;

#define _TO_STR ostringstream ostr;\
	ostr << value;\
	return ostr.str();

#endif //_COMMON_MACROS

#include <string>
#include <vector>
#include <sstream>
#include <sys/time.h>
#include <iostream>
#include <fstream>
#include <sys/stat.h>
#include <sys/time.h>
#include <dirent.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/wait.h>
#include <signal.h>
#include <cstring>
#include <algorithm>

#include "Datastore.h"
#include "Objectpath.h"

using namespace std;


/* ---------------------------------------------------------------------------*/
/*                                System Calls                                */
/* ---------------------------------------------------------------------------*/

// Execute a command and get the response
int CF_runCommand(const string& cmd, string& stdout_msg, string& stderr_msg, string& errorMessage);

// Execute a command and get the response. Set do_trace to trace commands (trace result in cmpi_prov_debug.txt)
int CF_runCommand(const string& cmd, string& stdout_msg, string& stderr_msg, bool do_trace, string& errorMessage);

// Execute a command and get the first line of the response
int CF_runCommandFL(const string& cmd, string& stdOut, string& errorMessage);

// Execute a command and split the response into lines.
// If expected_number_of_lines != 0 and the number of lines != expected_number_of_lines
// then a FAILED status is returned
int CF_runCommandToLines(const string& cmd, vector<string>& output, unsigned long expected_number_of_lines, string& errorMessage);

// Get the computer system name
int CF_getSystemName(string& sysName, string& errorMessage);

// Get the operating system name
int CF_getOSName(string& OSName, string& errorMessage);

//Get the list of ethernet port
//Deprecated
int CF_getEthernetPortNames(vector<string>& result, string& errorMessage);

//Get IP by the ethernet port name
//Deprecated
int CF_getIP(const string& ethName, string& ip, string& errorMessage);

//Get the ip of the first ethernet port.
int CF_getMachineIP(string& ip, string& errorMessage);

// Refer to the below URL for block size calculation
// http://www.faqs.org/docs/Linux-mini/Partition.html#BLOCKSIZE
int CF_getBlockSize(const string& path, int& blocksize, int& num_blocks, string& errorMessage);

// Get the first line of the response of "whatis" for a specified program name
int CF_getWhatisFL(const string& name, string& output, string& errorMessage);

/* ---------------------------------------------------------------------------*/
/*                             File manipulation                              */
/* ---------------------------------------------------------------------------*/

// Lock a file (multi-process safety)
int CF_lock(const struct flock& file_lock, int& fd, string& errorMessage);

// Unlock a file (multi-process safety)
int CF_unlock(struct flock& file_lock, int fd, string& errorMessage);

// Read the content of a file and put it into a string
int CF_readTextFile(const string& path, string& content, string& errorMessage);

// Read the content of a file and return the first line only
int CF_readTextFileFL(const string& path, string& content, string& errorMessage);

// Read the content of a file and split it into lines.
// If expected_number_of_lines != 0 and the number of lines != expected_number_of_lines
// then a FAILED status is returned
int CF_readTextFileToLines(const string& path, vector<string>& content, unsigned long expected_number_of_lines, string& errorMessage);

// The result of CF_writeTextFile(...) has unkonwn characters at the end.
// It remove those characters.
int CF_readFileToString(const string& path, string& result, string& errorMessage);

// Write the content of a string to a file
int CF_writeTextFile(const string& path, const string& content, string& errorMessage);

// Check whether or not the file exist.
bool CF_isExist(const string& filePath);

// Check if the given path corresponds to a regular file
bool CF_isRegularFile(const string& path);

// Check if the given path corresponds to a directory
bool CF_isDirectory(const string& path);

// List the regular files/directories/block devices/character devices contained in a directory
int CF_scanDirectory(const string& directory, vector<string>& regular_files, vector<string>& directories, vector<string>& block_devices, vector<string>& character_devices, string& errorMessage);


/* ---------------------------------------------------------------------------*/
/*                            String manipulation                             */
/* ---------------------------------------------------------------------------*/

// Remove desired characters from a string
string CF_removeChar(const string& input, const char* characters);

// Split a text into strings using one or multiple separators
void CF_splitText(vector<string>& output, const string& input, const char* separators);

// Split a text into strings
void CF_splitText(vector<string>& output, const string& input, char separator);

// Split a text into strings (space character as separator) and removes spaces
void CF_splitTextBySpace(vector<string>& output, const string& input);

// Remove the formatting characters (tabulations and backspaces)
string CF_trimText(const string& Text);

// Replace tabs by spaces in a string
string CF_untab(const string& text);

// Convert string to low case
string CF_toLowCase(const string& str);

// Check if "word" is at the begining of the string "str"
bool CF_startsWith(const string& str, const string& word);

// Check if "word" is at the begining of the string "str" (case insensitive version)
bool CF_startsWithNoCase(const string& str, const string& word);

// Check if "word" is at the end of the string "str"
bool CF_endsWith(const string& str, const string& word);

// Check if "word" is at the end of the string "str" (case insensitive version)
bool CF_endsWithNoCase(const string& str, const string& word);

// Add escape characters in front of double quotes.
// Useful for quoting strings
string CF_quoteString(const string& str);


/* ---------------------------------------------------------------------------*/
/*                            C++ type comparison                             */
/* ---------------------------------------------------------------------------*/

// Compare two strings in a case-insensitive manner
bool CF_strCmpNoCase(const string& str1, const string& str2);

// Compare two arrays of strings in a case-insensitive manner
bool CF_strArrayCmpNoCase(const vector<string>& vect1, const vector<string>& vect2);

// Compare two arrays of numerical values
bool CF_numArrayCmp(const vector<unsigned char>& vect1, const vector<unsigned char>& vect2);
bool CF_numArrayCmp(const vector<unsigned short>& vect1, const vector<unsigned short>& vect2);
bool CF_numArrayCmp(const vector<unsigned long>& vect1, const vector<unsigned long>& vect2);
bool CF_numArrayCmp(const vector<unsigned long long>& vect1, const vector<unsigned long long>& vect2);
bool CF_numArrayCmp(const vector<signed char>& vect1, const vector<signed char>& vect2);
bool CF_numArrayCmp(const vector<short>& vect1, const vector<short>& vect2);
bool CF_numArrayCmp(const vector<long>& vect1, const vector<long>& vect2);
bool CF_numArrayCmp(const vector<long long>& vect1, const vector<long long>& vect2);
bool CF_numArrayCmp(const vector<float>& vect1, const vector<float>& vect2);
bool CF_numArrayCmp(const vector<double>& vect1, const vector<double>& vect2);

// Compare two arrays of booleans
bool CF_boolArrayCmp(const vector<bool>& vect1, const vector<bool>& vect2);

// Compare two arrays of references
bool CF_refArrayCmp(const vector<Objectpath>& vect1, const vector<Objectpath>& vect2);

// Compare two string datetimes
// Result is given in sign:
// -1: datetime1 < datetime2
// 0: datetime1 = datetime2
// +1: datetime1 > datetime2
int CF_datetimeCmp(int& sign, const string& datetime1, const string& datetime2, string& errorMessage);


/* ---------------------------------------------------------------------------*/
/*                           CMPI type comparison                             */
/* ---------------------------------------------------------------------------*/

bool CF_CMPIDataCmp(const CMPIBroker* broker, const CMPIData& data1, const CMPIData& data2);


/* ---------------------------------------------------------------------------*/
/*                       Time and Datetime manipulation                       */
/* ---------------------------------------------------------------------------*/

// Get the number of minutes from the UTC timezone in minutes
short CF_getCurrentTimeZone();

// Get the time in seconds since the Epoch (UTC 1970.01.01)
time_t CF_getUTCTime();

// Append the timezone to a string following the CIM format for datetime properties
void CF_addTimeZone(string& datetime, short timezone);

// Convert a time_t C structure (time in seconds since the Epoch (UTC 1970.01.01))
// into a CIM datetime formated string without the timezone (local time)
string CF_toLocalTime(time_t time);

// Convert a time_t C structure (time in seconds since the Epoch (UTC 1970.01.01))
// into a CIM datetime formated string (local time)
string CF_toLocalTime(time_t UTC_time, short timezone);

// Get the local date time into a CIM datetime formated string
string CF_getLocalDateTime(time_t UTC_time);

// Get the last time when the state is changed
int CF_lastModified(const string& filepath, string& modifiedTime, string& errorMessage);

// Tells if the string contains a valid datetime or not
bool CF_isDatetime(const string& myDatetime);

// Converts a string datetime into its number of microseconds since the EPOCH (1970/01/01)
// Dates before the EPOCH are not supported
int CF_datetimeToBinary(unsigned long long& binaryDatetime, const string& myDatetime, string& errorMessage);

// Converts an abbreviated month name (month name on 3 characters) string to its number
int CF_monthAbvStrToInt(const string& month);

// Converts an abbreviated month name (month name on 3 characters) string (char* format) to its number
int CF_monthAbvStrToInt(const char* month);

// DEPRECATED
// Get the number of minutes from the UTC timezone in minutes (can be negative)
signed short CF_getOsTimezone();

// DEPRECATED
// Get the local time in seconds since the Epoch (1970.01.01)
time_t CF_localTime();

// DEPRECATED
// Append the timezone to a string following the CIM format for datetime properties
void CF_catTimezone(char *str, signed short zone);

// DEPRECATED
// Convert a time_t C structure into a CIM datetime formated string, including the timezone
string CF_timeToString(time_t time);


/* ---------------------------------------------------------------------------*/
/*                         Integer / String conversion                        */
/* ---------------------------------------------------------------------------*/

// Convert int to string
string CF_intToStr(unsigned char value);
string CF_intToStr(unsigned short value);
string CF_intToStr(unsigned long value);
string CF_intToStr(unsigned long long value);
string CF_intToStr(signed char value);
string CF_intToStr(short value);
string CF_intToStr(long value);
string CF_intToStr(long long value);
string CF_intToStr(float value);
string CF_intToStr(double value);
string CF_intToStr(int value);
string CF_intToStr(unsigned int value);

// convert boolean to string
string CF_boolToStr(bool value);

// Convert a string to an unsigned long
unsigned long CF_strToUL(const string& input);

// Convert a string to an unsigned long long
unsigned long long CF_strToULL(const string& input);

//Check if it is a number
bool CF_isNumber(const string& str);


/* ---------------------------------------------------------------------------*/
/*                                    Sorting                                 */
/* ---------------------------------------------------------------------------*/

// Find the index of an element in list
bool CF_foundInList(const string& element, const vector<string>& list, vector<string>::size_type& index);

// Find the index of an element in a sorted list
bool CF_foundInSortedList(const string& element, const vector<string>& list, vector<string>::size_type& index);


/* ---------------------------------------------------------------------------*/
/*                        Provider Debugging mechanism                        */
/* ---------------------------------------------------------------------------*/

// A simple debuging mechanism
// The debugging infomation is written in the /var/tmp/cmpi_prov_debug.txt file located where the cimserver was launched.
// _E_; // This will print "<< enter [FUNCTION]" when you enter a method.
// _L_; // This will print ">> leave [FUNCTION]" when you leave a method.
// DEBUG; // This will print "== debug [FILE][FUNCTION][LINE]"
#ifdef __DEBUG__
#define DEBUG system(((string) "echo \"== debug [" + CF_intToStr(getpid()) + "][" + __FILE__ + (string) "][" + __func__ + (string) "][" + CF_intToStr(__LINE__) + (string) "]\" >> /var/tmp/cmpi_prov_debug.txt").c_str())
#define _DEBUG(errorMessage) system(((string) "echo \"== debug [" + CF_intToStr(getpid()) + "][" + __FILE__ + (string) "][" + __func__ + (string) "][" + CF_intToStr(__LINE__) + (string) "]: " + errorMessage + (string) "\" >> /var/tmp/cmpi_prov_debug.txt").c_str())
#define _E_ system(((string) "echo \"<< enter [" + CF_intToStr(getpid()) + "][" + __func__ + (string) "]\" >> /var/tmp/cmpi_prov_debug.txt").c_str())
#define _L_ system(((string) "echo \">> leave [" + CF_intToStr(getpid()) + "][" + __func__ + (string) "]\" >> /var/tmp/cmpi_prov_debug.txt").c_str())
#else
#define DEBUG
#define _DEBUG(errorMessage)
#define _E_
#define _L_
#endif

#endif /*COMMON_H_*/
